﻿import mx.events.EventDispatcher;
import mx.utils.Delegate;

/**
 * This class can be used as a generic slider in your applications, may be as a voulme slider, a scrollbar or any other case where you have to use a slider.
 * The following events are dispatched by this class:<br/><br/>
 * 		<b><i>Scrolling</i></b> - This event keeps on firing from the moment the scrollface is pressed to the moment the scrollface is released. The event object has the following properties:<br/>
 * 			<i>position</i> - A number ranging from 1 to 100 that tells where the position of the scrollface is.<br/><br/>
 * 		<b><i>Scrolled</i></b> - This event is fired once the scrollface is released after the scrolling is done, or when the <i>scroll</i> or <i>scrollTo</i> functions are called. The event object has the following properties:<br/>
 * 			<i>position</i> - A number ranging from 1 to 100 that tells where the position of the scrollface is.
 * <br/><br/>
 * <b>Usage:</b>
 * 		In the movieclip you want to use as a scrollbar, create two movie clips with instance names <i>face_mc</i> and <i>base_mc</i>. Align the face_mc and base_mc movieclips properly. Then in the library, right-click on the movieclip that you want to use as scrollbar, and select <i>Linkage</i>. In the <i>Class:</i> field, write <i>com.maheshj.UI.scroller.ScrollBar</i> and click <i>OK</i>. Make sure that the <i>com</i> directory in which the scrollbar classes are provided are available in the FLA's classpaths.<br/>
 */
class com.maheshj.UI.ScrollBar extends MovieClip {
	
	private static var EventDispatcherDependancy = EventDispatcher.initialize(ScrollBar.prototype);
	
	private var faceHitPoint:Number;
	private var scrollid:Number;
	private var orig_base_height:Number;
	private var orig_face_height:Number;
	
	private var face_mc:MovieClip;
	private var base_mc:MovieClip;
	
	private function ScrollBar() {
		init();
	}
	
	/**
	 * This function initializes the scrollbar. It records the height of the base and face mc's and sets up scroll events.
	 */
	private function init() {
		
		orig_base_height = base_mc._height;
		orig_face_height = face_mc._height;
		
		face_mc._y = base_mc._y;
		
		face_mc.onMouseDown = Delegate.create(this, function() {
			if ((_xmouse >= face_mc._x) and (_xmouse <= face_mc._x + face_mc._width) and (_ymouse >= face_mc._y) and (_ymouse <= face_mc._y + face_mc._height)) {
				drag();
			}
			
		});
		face_mc.onMouseUp = Delegate.create(this, function() {
			clearInterval(scrollid);
			dispatchEvent( { type:"Scrolled", position:Math.round((face_mc._y/(scrollHeight - faceHeight))*100)} );
		});
		base_mc.onMouseDown = Delegate.create(this, function() {
			if ((_xmouse >= base_mc._x) and (_xmouse <= base_mc._x + base_mc._width) and (_ymouse >= base_mc._y) and (_ymouse <= base_mc._y + base_mc._height)) {
				scrollTo(_ymouse - faceHeight / 2 );
			}
		});
		this.addEventListener("Scrolled", this);
		this.addEventListener("Scrolling", this);
	}
	
	/**
	 * This method is called when the face mc is pressed. It calls "updateFacePosition" at an interval of 1 ms.
	 */
	private function drag() {
		faceHitPoint = (face_mc._ymouse/orig_face_height)*faceHeight;
		scrollid = setInterval(this, "updateFacePosition" ,1);
	}
	
	/**
	 * This method is called at an interval of 1 ms to update the position of the face mc based on mouse position.
	 */
	private function updateFacePosition() {
		if((_ymouse >= faceHitPoint) and (_ymouse <= scrollHeight - (faceHeight - faceHitPoint))){
			face_mc._y = _ymouse - faceHitPoint;
		}
		if (_ymouse < faceHitPoint) {
			face_mc._y = 0;
		}
		if (_ymouse > scrollHeight - (faceHeight - faceHitPoint)) {
			face_mc._y = scrollHeight - faceHeight;
		}
		updateAfterEvent();
		dispatchEvent( { type:"Scrolling", position:Math.round((face_mc._y/(scrollHeight - faceHeight))*100), scrollHeight:scrollHeight} );
	}
	
	/**
	 * If you want to implement a custom scroll, <i>i.e.,</i> want to scroll the scrollbar by a certain amount from outside the scrollbar without using the scrollface, then you can use this method.
	 * @param	num - a parameter of type Number that denotes the amount by which it is to be scrolled.
	 */
	public function scroll(num:Number) {
		face_mc._y += num;
		if (face_mc._y > scrollHeight - faceHeight) {
			face_mc._y = scrollHeight - faceHeight;
		}
		if (face_mc._y < 0) {
			face_mc._y = 0;
		}
		dispatchEvent( { type:"Scrolled", position:Math.round((face_mc._y/(scrollHeight - faceHeight))*100)} );
	}
	
	/**
	 * If you want to implement a custom scroll, <i>i.e.,</i> want to scroll the scrollbar to a certain position from outside the scrollbar without pressing the scrollface, then you can use this method.
	 * @param	num - a parameter of type Number that denotes the position to which it is to be scrolled.
	 */
	public function scrollTo(num:Number) {
		if (num != undefined) {
			face_mc._y = num;
			if (face_mc._y > scrollHeight - faceHeight) {
				face_mc._y = scrollHeight - faceHeight;
			}
			if (face_mc._y < 0) {
				face_mc._y = 0;
			}
			dispatchEvent( { type:"Scrolled", position:Math.round((face_mc._y/(scrollHeight - faceHeight))*100)} );
		}
	}
	
	public function addEventListener() {}
	
	public function removeEventListener() {}
	
	private function dispatchEvent() {}
	
	
	/**
	 * Use this property to get and set the height of the scrollbase.
	 */
	public function get scrollHeight():Number {
		return base_mc._height;
	}
	
	public function set scrollHeight(num:Number) {
		if (num != undefined) {
			face_mc._y = (num / base_mc._height) * face_mc._y;
			base_mc._height = num;
			if (face_mc._y > scrollHeight - faceHeight) {
				face_mc._y = scrollHeight - faceHeight;
			}
			dispatchEvent( { type:"Scrolling", position:Math.round((face_mc._y/(scrollHeight - faceHeight))*100)} );
		}else {
			base_mc._height = 100;
		}
	}
	
	/**
	 * Use this property to get and set the height of the scrollface.
	 */
	public function get faceHeight():Number {
		return face_mc._height;
	}
	
	public function set faceHeight(num:Number) {
		if ((num < base_mc._height) and (num > 0)) {
			face_mc._height = num;
		}else {
			face_mc._height = orig_face_height
		}
		scroll();
	}
}