﻿import com.maheshj.UI.ScrollBar;
import com.maheshj.utils.ContentLoader;
import mx.events.EventDispatcher;
import mx.utils.Delegate;
import caurina.transitions.*;
import com.pixelbreaker.ui.MouseWheel;

/**
 * This is a scroller class into which you can load jpg, png, gif or swf's. You can scroll the content both horizontally and vertically. The class will take care of all the resizing for you and all you have to do is initialize it with the desired scrolling dimensions. You may change them later as well. The following events are dispatched by this class:<br/><br/>
 * 		<b><i>ScrollPaneResized</i></b> - This event is fired when the scroller is resized from the outside (maybe due to a stage resize listener). The event object has the following properties:<br/>
 * 			<i>width</i> - A number denoting the width of the displayed content.<br/>
 * 			<i>height</i> - A number denoting the height of the displayed content.<br/>
 * 		You can listen to this event to know when the scrollbox is resized. The scrollbox has a movieclip with name content_mc into which all the dynamic content will be loaded. This movieclip is registered with the event by default when the scrollbox is initialized.<br/><br/>
 * 		<b><i>ConLoadStarted</i></b> - Same as <i>onLoadStart</i> event of <i>MovieClipLoader</i> class. Fired when content starts loading after <i>loadContent</i> function is called.<br/><br/>
 * 		<b><i>ConLoading</i></b> - Same as <i>onLoadProgress</i> event of <i>MovieClipLoader</i> class. Fired when content progresses loading after <i>loadContent</i> function is called. The event object has an object names <i>loadProgress</i> which has two properties <i>bytesLoaded</i> and <i>bytesTotal</i> which can be used to track the progress of the loading.<br/><br/>
 * 		<b><i>ConLoadInit</i></b> - Same as <i>onLoadInit</i> event of <i>MovieClipLoader</i> class. Fired when content initializes loading after <i>loadContent</i> function is called.<br/><br/>
 * 		<b><i>ConLoadComplete</i></b> - Same as <i>onLoadComplete</i> event of <i>MovieClipLoader</i> class. Fired when content completes loading after <i>loadContent</i> function is called.<br/><br/>
 * 		<b><i>ConLoadError</i></b> - Same as <i>onLoadError</i> event of <i>MovieClipLoader</i> class. Fired when error occurs while loading content after <i>loadContent</i> function is called.<br/><br/>
 * 
 * <b>Usage:</b>
 * 		Since this class extends MovieClip class, you must attach it to a movieclip that you want to behave as a scrollbox. For this class to function properly, you need to have <i>mask_mc</i> movieclip that will mask the content to be scrolled, two scrollbars, <i>hscroll_mc</i> and <i>vscroll_mc</i> which must be instances of ScrollBar class. See the documentation of <i>ScrollBar</i> class to see how to make a movieclip that can behave as a scrollbar. First make a vertical scrollbar and put it in the scroller movieclip. Then duplicate it in the library. Give it a different name. Put it inside the scroller movieclip and rotate it counter-clockwise 90 degress to make a horizontal scrollbar. Then flip it vertically to bring its registration point to the top-left. You also need to have a movieclip named <i>content_mc</i>. The content will be loaded into this movieclip. All this movieclips must be placed inside the movieclip to which you will attach this class.
 * 
 * To attach this class to a movieclip, in the library, right-click on the movieclip that you want to use as scrollbox, and select <i>Linkage</i>. In the <i>Class:</i> field, write <i>com.maheshj.UI.scroller.ScrollBox</i> and click <i>OK</i>. Make sure that the <i>com</i> directory in which the scrollbar classes are provided are available in the FLA's classpaths.<br/>
 */
class com.maheshj.UI.ScrollBox extends MovieClip {
	
	private static var EventDispatcherDependancy = EventDispatcher.initialize(ScrollBox.prototype);
	
	private var mask_mc:MovieClip;
	public var content_mc:MovieClip;
	private var con_cl:ContentLoader;
	private var hscroll_mc:ScrollBar;
	private var vscroll_mc:ScrollBar;
	
	private var h_scroll:Boolean;
	private var v_scroll:Boolean;
	private var scroll_scale:Boolean;
	
	private var scrollWidth:Number;
	private var scrollHeight:Number;
	private var orig_scrollWidth:Number;
	private var orig_scrollHeight:Number;
	
	private var mouseListener:Object;
	private var keyListener:Object;
	
	private var mousefactor:Number;
	private var keyfactor:Number;
	private var scroll_time:Number;
	private var scroll_delay:Number;
	
	private var scroll_ease:String;
	
	private function ScrollBox() {
		onEnterFrame = Delegate.create(this, function() {
			if (scrollsHorizontally == undefined) {
				init(mask_mc._width, mask_mc._height, true, true);
			}
		});
	}
	
	/**
	 * This method is used to initialize the scrollbox with the desired properties as listed below. All these parameters are except <i>hScrollHeight</i> and <i>vScrollHeight</i> are optional. If others parameters are not specified in the function call, they are initialized to <i>false</i>
	 * @param	hScrollHeight - Number representing the height of the horizontal scrollbar as well as its scrollheight
	 * @param	vScrollHeight - Number representing the height of the vertical scrollbar as well as its scrollheight
	 * @param	hScroll - <i>[Optional]</i> A boolean value to specify whether horizontal scrolling is enabled or not.
	 * @param	vScroll - <i>[Optional]</i> A boolean value to specify whether vertical scrolling is enabled or not.
	 * @param 	mouse - <i>[Optional]</i> A boolean value to specify whether mouse wheel scrolling is enabled or not.
	 * @param	keyboard - <i>[Optional]</i>A boolean value to specify whether scrolling with keyboard is enabled or not.
	 * @param	scaleBars - <i>[Optional]</i> A boolean value to specify whether the faces of the scrollbar will be scaled according to the content dimensions.
	 */
	public function init(hScrollHeight:Number, vScrollHeight:Number, hScroll:Boolean, vScroll:Boolean, mouse:Boolean, keyboard:Boolean, scaleBars:Boolean) {
		
		this.addEventListener("ScrollPaneResized", content_mc);
		
		hscroll_mc.addEventListener("Scrolling", Delegate.create(this, scrollHorizontal));
		vscroll_mc.addEventListener("Scrolling", Delegate.create(this, scrollVertical));
		hscroll_mc.addEventListener("Scrolled", Delegate.create(this, scrollHorizontal));
		vscroll_mc.addEventListener("Scrolled", Delegate.create(this, scrollVertical));
		
		con_cl = new ContentLoader();
		
		con_cl.onLoadStart = Delegate.create(this, function() {
			hscroll_mc.scrollTo(0);
			vscroll_mc.scrollTo(0);
			resize();
			dispatchEvent( { type:"ConLoadStarted" } );
		});
		con_cl.onLoadInit = Delegate.create(this, function() {
			resize();
			dispatchEvent( { type:"ConLoadInit" } );
		});
		con_cl.onLoadComplete = Delegate.create(this, function() {
			dispatchEvent( { type:"ConLoadComplete" } );
		});
		con_cl.onLoadProgress = Delegate.create(this, function(t_mc:MovieClip, bl:Number, bt:Number) {
			dispatchEvent( { type:"ConLoading", loadProgress:con_cl.getProgress(t_mc) } );
		});
		con_cl.onLoadError = Delegate.create(this, function(t_mc:MovieClip, errorCode:String, httpStatus:Number) {
			dispatchEvent( { type:"ConLoadError", errorCode:errorCode, httpStatus:httpStatus } );
		});
		
		mouseWheelSpeed = 5;
		keyBoardSpeed = 15;
		scrollDelay = 0;
		scrollTime = 0.5;
		scrollEaseType = "easeOutQuad"
		
		scrollsHorizontally = hScroll;
		scrollsVertically = vScroll;
		mouseScrollEnabled = mouse;
		keyBoardScrollEnabled = keyboard;
		scaleScrollBars = scaleBars;
		
		resize(hScrollHeight, vScrollHeight);
		
	}
	
	/**
	 * If the ScrollBox is used to load content dynamically into it at runtime, then the content can be loaded with this method.
	 * @param	path - The path of the file to be loaded into <i>content_mc</i> movieclip.
	 */
	public function loadContent(path:String) {
		con_cl.loadContent(path, content_mc);
	}
	
	/**
	 * If you want to resize the scrollbox at runtime after setting up the scroll width and height initially, you can use this method.
	 * @param	width - <i>[Optional]</i> The width to which it is to be resized.
	 * @param	height - <i>[Optional]</i> The height to which it is to be resized.
	 */
	public function resize(width:Number, height:Number) {
			if ((width != undefined) and (width > 0)) {
				orig_scrollWidth = width;
				orig_scrollHeight = height;
				if (scrollsVertically == true) {
					if (content_mc._height > vscroll_mc.scrollHeight) {
						scrollWidth = width - vscroll_mc._width;
					}else {
						scrollWidth = width
					}
				}else {
					scrollWidth = width
				}
			}else {
				if (scrollsVertically == true) {
					if (content_mc._height > vscroll_mc.scrollHeight) {
						scrollWidth = orig_scrollWidth - vscroll_mc._width;
					}else {
						scrollWidth = orig_scrollWidth;
					}
				}else {
					scrollWidth = orig_scrollWidth;
				}
			}
		
			if ((height != undefined) and (height > 0)) {
				if (scrollsHorizontally == true) {
					if (content_mc._width > hscroll_mc.scrollHeight) {
						scrollHeight = height - hscroll_mc._height;
					}else {
						scrollHeight = height;
					}
				}else {
					scrollHeight = height;
				}
			}else {
				if (scrollsHorizontally == true) {
					if (content_mc._width > hscroll_mc.scrollHeight) {
						scrollHeight = orig_scrollHeight - hscroll_mc._height;
					}else {
						scrollHeight = orig_scrollHeight;
					}
				}else {
					scrollHeight = orig_scrollHeight;
				}
			}
		
			hscroll_mc.scrollHeight = scrollWidth;
			vscroll_mc.scrollHeight = scrollHeight;
		
			mask_mc._width = scrollWidth;
			mask_mc._height = scrollHeight;
		
			dispatchEvent( { type:"ScrollPaneResized" , width:hscroll_mc.scrollHeight, height:vscroll_mc.scrollHeight} );
		
			arrangeScrollBars();
	}
	
	private function arrangeScrollBars() {
		if ((content_mc._width != 0) and (content_mc._height != 0)) {
			if (scaleScrollBars) {
				if (Math.round((hscroll_mc.scrollHeight/content_mc._width)*hscroll_mc.scrollHeight) < 30) {
					hscroll_mc.faceHeight = 30;
				}else {
					hscroll_mc.faceHeight = Math.round((hscroll_mc.scrollHeight/content_mc._height)*hscroll_mc.scrollHeight);
				}
				if (Math.round((vscroll_mc.scrollHeight/content_mc._height)*vscroll_mc.scrollHeight) < 30) {
					vscroll_mc.faceHeight = 30;
				}else {
					vscroll_mc.faceHeight = Math.round((vscroll_mc.scrollHeight/content_mc._height)*vscroll_mc.scrollHeight);
				}
			}else {
				hscroll_mc.faceHeight = 0;
				vscroll_mc.faceHeight = 0;
			}
			
			switchScrollBarsOn();
			
			hscroll_mc._x = 0;
			hscroll_mc._y = scrollHeight;
			
			vscroll_mc._x = scrollWidth;
			vscroll_mc._y = 0;
		}else {
			switchScrollBarsOff();
		}
	}
	
	private function switchScrollBarsOn() {
		if (scrollsHorizontally == true) {
			if (content_mc._width > hscroll_mc.scrollHeight) {
				hscroll_mc._visible = true;
			}else {
				hscroll_mc._visible = false;
			}
		}else {
			hscroll_mc._visible = false;
		}
		if (scrollsVertically == true) {
			if (content_mc._height > vscroll_mc.scrollHeight) {
				vscroll_mc._visible = true;
			}else{
				vscroll_mc._visible = false;
			}
		}	
	}
	
	private function switchScrollBarsOff() {
		hscroll_mc._visible = false;
		vscroll_mc._visible = false;
		
	}
	
	private function scrollHorizontal(e:Object) {
		Tweener.addTween(content_mc, {_x:-(Math.round((content_mc._width - e.target.scrollHeight)*e.position/100)), time:scrollTime, delay:scrollDelay, transition:scrollEaseType});
	}
	
	private function scrollVertical(e:Object){
		Tweener.addTween(content_mc, {_y:-(Math.round((content_mc._height - e.target.scrollHeight)*e.position/100)), time:scrollTime, delay:scrollDelay, transition:scrollEaseType});
	}
	
	private function setupMouseWheel() {
		mouseListener = new Object();
		
		mouseListener.onMouseWheel = Delegate.create(this, function(delta) {
			if ((mask_mc._xmouse >= 0) and (mask_mc._xmouse <= mask_mc._width) and (mask_mc._ymouse >= 0) and (mask_mc._ymouse <= mask_mc._height)) {
				if (Key.isDown(Key.CONTROL) == true) {
					if (scrollsHorizontally == true) {
						if (content_mc._width > hscroll_mc.scrollHeight) {
							hscroll_mc.scroll(-mousefactor*delta);
						}
					}
				}else {
					if (Key.isDown(Key.SHIFT) == false) {
						if (scrollsVertically == true) {
							if (content_mc._height > vscroll_mc.scrollHeight) {
								vscroll_mc.scroll(-mousefactor*delta);
							}
						}else {
							if (scrollsHorizontally == true) {
								if (content_mc._width > hscroll_mc.scrollHeight) {
									hscroll_mc.scroll(-mousefactor*delta);
								}
							}
						}
					}
				}
				if (Key.isDown(Key.SHIFT) == true) {
					if (scrollsHorizontally == true) {
						if (content_mc._width > hscroll_mc.scrollHeight) {
							hscroll_mc.scroll(-mousefactor*delta);
						}
					}
					if (scrollsVertically == true) {
						if (content_mc._height > vscroll_mc.scrollHeight) {
							vscroll_mc.scroll(-mousefactor*delta);
						}
					}
				}
			}
		});
		
		//Mouse.addListener(mouseListener);
		MouseWheel.addListener(mouseListener);
	}
	
	private function setupKeyBoard() {
		keyListener = new Object();
		keyListener.onKeyDown = Delegate.create(this, function() {
			if (Key.isDown(Key.UP)) {
				if (scrollsVertically == true) {
					if (content_mc._height > vscroll_mc.scrollHeight) {
						vscroll_mc.scroll(-keyfactor);
					}
				}
			}
			if (Key.isDown(Key.DOWN)) {
				if (scrollsVertically == true) {
					if (content_mc._height > vscroll_mc.scrollHeight) {
						vscroll_mc.scroll(keyfactor);
					}
				}
			}
			if (Key.isDown(Key.HOME) and !Key.isDown(Key.CONTROL)) {
				if (scrollsVertically == true) {
					if (content_mc._height > vscroll_mc.scrollHeight) {
						vscroll_mc.scrollTo(0);
					}
				}else {
					if (scrollsHorizontally == true) {
						if (content_mc._width > hscroll_mc.scrollHeight) {
							hscroll_mc.scrollTo(0);
						}
					}
				}
			}
			if (Key.isDown(Key.END) and !Key.isDown(Key.CONTROL)) {
				if (scrollsVertically == true) {
					if (content_mc._height > vscroll_mc.scrollHeight) {
						vscroll_mc.scrollTo(vscroll_mc.scrollHeight);
					}
				}else {
					if (scrollsHorizontally == true) {
						if (content_mc._width > hscroll_mc.scrollHeight) {
							hscroll_mc.scrollTo(hscroll_mc.scrollHeight);
						}
					}
				}
			}
			if (Key.isDown(Key.RIGHT)) {
				if (scrollsHorizontally == true) {
					if (content_mc._width > hscroll_mc.scrollHeight) {
						hscroll_mc.scroll(keyfactor);
					}
				}
			}
			if (Key.isDown(Key.LEFT)) {
				if (scrollsHorizontally == true) {
					if (content_mc._width > hscroll_mc.scrollHeight) {
						hscroll_mc.scroll(-keyfactor);
					}
				}
			}
			if (Key.isDown(Key.CONTROL) and Key.isDown(Key.HOME)) {
				if (scrollsHorizontally == true) {
					if (content_mc._width > hscroll_mc.scrollHeight) {
						hscroll_mc.scrollTo(0);
					}
				}
			}
			if (Key.isDown(Key.CONTROL) and Key.isDown(Key.END)) {
				if (scrollsHorizontally == true) {
					if (content_mc._width > hscroll_mc.scrollHeight) {
						hscroll_mc.scrollTo(hscroll_mc.scrollHeight);
					}
				}
			}
			if (Key.isDown(Key.SHIFT) and Key.isDown(Key.HOME)) {
				if (scrollsHorizontally == true) {
					if (content_mc._width > hscroll_mc.scrollHeight) {
						hscroll_mc.scrollTo(0);
					}
				}
				if (scrollsVertically == true) {
					if (content_mc._height > vscroll_mc.scrollHeight) {
						vscroll_mc.scrollTo(0);
					}
				}
			}
			if (Key.isDown(Key.SHIFT) and Key.isDown(Key.END)) {
				if (scrollsHorizontally == true) {
					if (content_mc._width > hscroll_mc.scrollHeight) {
						hscroll_mc.scrollTo(hscroll_mc.scrollHeight);
					}
				}
				if (scrollsVertically == true) {
					if (content_mc._height > vscroll_mc.scrollHeight) {
						vscroll_mc.scrollTo(vscroll_mc.scrollHeight);
					}
				}
			}
		});
		Key.addListener(keyListener);
	}
	
	public function addEventListener() {}
	
	public function removeEventListener() {}
	
	private function dispatchEvent() { }
	
	/**
	 * Use this property to get or set whether the scrollbox will scroll horizontally or not. The valid values are <i>true</i> or <i>false</i>
	 */
	public function get scrollsHorizontally():Boolean {
		return h_scroll;
	}
	
	public function set scrollsHorizontally(bool:Boolean) {
		h_scroll = bool;
		if (h_scroll) {
			hscroll_mc._visible = true;
		}else {
			hscroll_mc._visible = false;
		}
		resize();
	}
	
	/**
	 * Use this property to get or set whether the scrollbox will scroll vertically or not. The valid values are <i>true</i> or <i>false</i>
	 */
	public function get scrollsVertically():Boolean {
		return v_scroll;
	}
	public function set scrollsVertically(bool:Boolean) {
		v_scroll = bool;
		if (v_scroll) {
			vscroll_mc._visible = true;
		}else {
			vscroll_mc._visible = false;
		}
		resize();
	}
	
	/**
	 * Use this property to get or set whether the faces of the scrollbars scale according to the content dimensions or not. The valid values are <i>true</i> or <i>false</i>
	 */
	public function get scaleScrollBars():Boolean {
		return scroll_scale;
	}
	
	public function set scaleScrollBars(bool:Boolean) {
		if (bool != undefined) {
			scroll_scale = bool;
		}else {
			scroll_scale = false;
		}
		arrangeScrollBars();
	}
	
	/**
	 * Use this property to enable or disable mousewheel scroll for the scrollbox. Valid values are <i>true</i> and <i>false</i>
	 */
	public function get mouseScrollEnabled():Boolean {
		if (mouseListener == undefined) {
			return false;
		}else {
			return true;
		}
	}
	
	public function set mouseScrollEnabled(bool:Boolean) {
		if (bool) {
			setupMouseWheel();
		}else {
			Mouse.removeListener(mouseListener);
			delete mouseListener;
		}
	}
	
	/**
	 * Use this property to enable or disable keyboard scroll for the scroller.  Valid values are <i>true</i> and <i>false</i>
	 */
	public function get keyBoardScrollEnabled():Boolean {
		if (keyListener == undefined) {
			return false;
		}else {
			return true;
		}
	}
	
	public function set keyBoardScrollEnabled(bool:Boolean) {
		if (bool) {
			setupKeyBoard();
		}else {
			Key.removeListener(keyListener);
			delete keyListener;
		}
	}
	
	/**
	 * Use this property to get or set the extent of scrolling that happens with the mouse wheel. Default value is 5. You can use any number greater than zero.
	 */
	public function get mouseWheelSpeed() {
		return mousefactor;
	}
	
	public function set mouseWheelSpeed(num:Number) {
		if (num != undefined) {
			mousefactor = Math.abs(num);
		}
	}
	
	/**
	 * Use this property to get or set the extent of scrolling that happens with the keyboard. Default value is 15. You can use any number greater than zero.
	 */
	public function get keyBoardSpeed():Number {
		return keyfactor;
	}
	
	public function set keyBoardSpeed(num:Number) {
		if (num != undefined) {
			keyfactor = Math.abs(num);
		}
	}
	
	/**
	 * Use this property to get and set the delay in scrolling. Default value is 0.
	 */
	public function get scrollDelay() {
		return scroll_delay;
	}
	
	public function set scrollDelay(num:Number) {
		if (num != undefined) {
			scroll_delay = Math.abs(num);
		}
	}
	
	/**
	 * Use this property to get and set the time for the scrolling to happen. Default value is 0.5.
	 */
	public function get scrollTime() {
		return scroll_time;
	}
	
	public function set scrollTime(num:Number) {
		if (num != undefined) {
			scroll_time = Math.abs(num);
		}
	}
	
	/**
	 * Use this property to get and set the easing type for the scrolling animation. Default value is <i>easeOutQuad</i>. But you can use all the easeing types offered by Tweener.
	 */
	public function get scrollEaseType() {
		return scroll_ease;
	}
	
	public function set scrollEaseType(type:String) {
		scroll_ease = type;
	}
}